<?php
namespace bot\generalClassBuilder;

require_once __DIR__ . '/generalFunctions.php';
require_once __DIR__ . '/ibsng/IBSngHandlers.php';

use bot\generalFunctions\generalFunctions;
use bot\generalDBFunctions\generalDBFunctions;

/**
 * Final, lazy, and minimal.
 * - No heavy work in constructor.
 * - Per-request singleton instances from getters.
 * - Shared config hydrated once (fast via APCu).
 */
class generalClassBuilder
{
    /** @var generalFunctions|null */
    private ?generalFunctions $gf = null;

    /** @var generalDBFunctions|null */
    private ?generalDBFunctions $gdb = null;

    /** @var \IBSngHandlers|null */
    private ?\IBSngHandlers $ibs = null;

    /** @var array|null */
    private ?array $cfg = null;

    public function __construct()
    {
        // Do nothing heavy here.
        // Let getters initialize on demand.
    }

    /** Fast, cached config shared by all dependencies */
    private function getSharedConfig(): array
    {
        if ($this->cfg !== null) return $this->cfg;

        // Try APCu (5–10x faster than re-reading files/DB each request)
        $key = 'bot.shared_cfg.v1';
        $hit = false;
        if (function_exists('apcu_fetch')) {
            $cached = apcu_fetch($key, $hit);
            if ($hit && is_array($cached)) {
                return $this->cfg = $cached;
            }
        }

        // Build config ONCE (keep pure PHP, no I/O here if possible)
        // If you currently read from DB/files, move that OUT of constructors
        // and keep it here; or make those readers APCu-cache their outputs too.
        $cfg = [
            // Example of deriving constants once; adjust to your real needs:
            'ibs_conn' => defined('IBSngConnection') ? IBSngConnection : null,
            // 'env' => $_ENV + $_SERVER (optionally pick what you need),
            // 'static_opts' => [...],
        ];

        if (function_exists('apcu_store')) {
            // TTL optional (e.g., 300s). Omit to cache until APCu eviction.
            apcu_store($key, $cfg, 300);
        }

        return $this->cfg = $cfg;
    }

    /** Lazy & cached per-request */
    public function getGeneralFunctions(): generalFunctions
    {
        // If generalFunctions currently hits DB or disk in __construct(),
        // refactor that class to accept pre-hydrated config from getSharedConfig()
        // and defer any I/O until actually needed.
        return $this->gf ??= new generalFunctions(/* $this->getSharedConfig() */);
    }

    /** Lazy & cached per-request */
    public function getGeneralDBFunctions(): generalDBFunctions
    {
        // Consider enabling PDO persistent connections *inside* this class
        // (PDO::ATTR_PERSISTENT => true) to avoid connect cost per request.
        return $this->gdb ??= new generalDBFunctions(/* $this->getSharedConfig() */);
    }

    /** Lazy & cached per-request */
    public function getIbsngHandlers(): \IBSngHandlers
    {
        $cfg = $this->getSharedConfig();
        // Avoid passing a constant directly if constructing does I/O.
        // Prefer a plain array from cached config.
        return $this->ibs ??= new \IBSngHandlers($cfg['ibs_conn']);
    }
}
